package com.blogspot.m3g4h4rd.wicket;

import java.util.HashSet;
import java.util.List;

import org.apache.wicket.AttributeModifier;
import org.apache.wicket.Component;
import org.apache.wicket.Session;
import org.apache.wicket.behavior.IBehavior;
import org.apache.wicket.behavior.SimpleAttributeModifier;
import org.apache.wicket.feedback.ContainerFeedbackMessageFilter;
import org.apache.wicket.feedback.FeedbackMessage;
import org.apache.wicket.feedback.IFeedback;
import org.apache.wicket.feedback.IFeedbackMessageFilter;
import org.apache.wicket.markup.html.WebMarkupContainer;
import org.apache.wicket.markup.html.form.Radio;
import org.apache.wicket.model.Model;

/**
 * This feedback component provides a way to give a validation feedback for one or 
 * more fields (wicket form based validation) involved by using CSS classes.
 * It causes a given set of child components (components contained directly or indirectly in
 * the feedback container) to show a feedback in the case a validation error occurs
 * at any child component. Moreover, a validation feedback container allows the user 
 * to define the set of child components that are observed as a source of validation errors.<br/>  
 * 
 * The default behavior is that all components in the container will visualize an error
 * feedback in the case any of the child components produces a validation error.
 * The set of components that will visualize a validation error,
 * as well as the components that may be observed as a source of validation error may 
 * be restricted:<br/>
 * <br/>
 * 
 * The method {@link #addErrorFeedbackComponentId(String)} allows to add a given id of a compoent
 * to a list of components that will visualize the error feedback. Only those
 * components whose id in this list will visualize a feedback. <br/>
 * <br/>
 * 
 * The method {@link #addFeedbackExcludedComponentId(String)} does the opposite of the 
 * method above, i.e., it excludes the component with the given id from the set of components
 * that visualize an error feedback. Note that you should use either 
 * {@link #addErrorFeedbackComponentId(String)} or  
 * {@link #addFeedbackExcludedComponentId(String)} but not both methods. If no component id 
 * is added with one of the two methods above, all child components visualize the error feedback.
 * <br/><br/>
 * 
 * The method {@link #addIgnoredComponentId(String)}
 * allows to define components whose error messages are ignored for giving
 * feedback. <br/>
 * 
 * The method {@link #addCheckedComponentId(String)} does the opposite of the method 
 * {@link #addIgnoredComponentId(String)}, i.e., it adds an id of a component whose 
 * validation error messages will be shown by the feedback container. Note that you should use either 
 * {@link #addCheckedComponentId(String)} or  
 * {@link #addIgnoredComponentId(String)} but not both methods. If no component id 
 * is added with one of the two methods above, all child components are observed as a 
 * source of validation errors.
 * 
 * There are further methods which allow to remove component and add more than one component
 * id at a time from the described sets of compoenent ids. 
 * 
 * Note that some parts of this component are based on 
 * {@link org.apache.wicket.markup.html.form.validation.FormComponentFeedbackBorder}.
 * 
 * @author Silvio Meier
 * 
 */
public class CSSFeedbackContainer extends WebMarkupContainer implements IFeedback {
	/**
	 * Serial version.
	 */
	private static final long serialVersionUID = 1L;
	
	/**
	 * The name of the css class containing the error feedback attribute. This class name
	 * is added as css class attribute in components that contain an error.
	 */
	private String cssClassName = "";

	/**
	 * The components whose id is in this set will not visualize the error feedback.
	 * The set of excluded components has precedence before the the set included components.
	 */
	private HashSet feedbackExcludedComponentIds;
	
	/**
	 * Only the components whose ids are in this set will show an 
	 * error feedback. If this set and {@link #feedbackExcludedComponentIds} is empty, 
	 * all components show the feedback.
	 */
	private HashSet errorFeedbackComponentsIds;
	
	/**
	 * Contains a list of ids of components whose error messages will be ignored 
	 * as a source of validation error by this feedback component.
	 */
	private HashSet ignoredComponentIds;
	
	/**
	 * Contains a list of ids of components which are the only components in the container
	 * whose validation error messages are checked. If this set and {@link #ignoredComponentIds} 
	 * is empty, all components may be the source of a validation error.
	 */
	private HashSet checkedComponentIds;
	
	/**
	 * Constructor.
	 * 
	 * @param id See {@link org.apache.wicket.Component}. The wicket id with which this component
	 * is bound to. The corresponding tag in the HTML source may be for example a <i>&lt;wicket:container&gt;</i> 
	 * tag.
	 * @param cssClassName The css class which represents an element in the html visualizing a 
	 * feedback error.
	 */
	public CSSFeedbackContainer(final String id, String cssClassName)
	{
		super(id);
		this.cssClassName = cssClassName;
		errorFeedbackComponentsIds = new HashSet();
		ignoredComponentIds = new HashSet();
		feedbackExcludedComponentIds = new HashSet();
		checkedComponentIds = new HashSet();
	}

	/**
	 * 
	 * @param id See {@link org.apache.wicket.Component}. See {@link org.apache.wicket.Component}. The wicket id with which this component
	 * is bound to. The corresponding tag in the HTML source may be for example a <i>&lt;wicket:container&gt;</i> 
	 * tag.
	 * @param cssClassName The css class which represents an element in the html visualizing a 
	 * feedback error.
	 * @param ids An array of ids that will visualize any observed validation in the feedback container. 
	 */
	public CSSFeedbackContainer(final String id, String cssClassName, String[] ids) {
		this(id, cssClassName);
		for (String i : ids) {
			errorFeedbackComponentsIds.add(i);
		}
	}
	
	/**
	 * 
	 * @param id See {@link org.apache.wicket.Component}. See {@link org.apache.wicket.Component}. The wicket id with which this component
	 * is bound to. The corresponding tag in the HTML source may be for example a <i>&lt;wicket:container&gt;</i> 
	 * tag.
	 * @param cssClassName The css class which represents an element in the html visualizing a 
	 * feedback error.
	 * @param ids An array of ids that will visualize any observed validation in the feedback container.
	 * @param ignoreIds An array of ids of components that will not be the source of an error.
	 */
	public CSSFeedbackContainer(final String id, String cssClassName, String[] ids, String[] ignoreIds) {
		this(id, cssClassName, ids);
		for (String i : ignoreIds) {
			ignoredComponentIds.add(i);
		}
	}
	
	/**
	 * Method that renders this component. The rendering procedure selects all direct or indirect child 
	 * components that cause a validation error and whose messages are not ignored by this 
	 * error feedback container. Any found validation errors are shown in the child components that are
	 * defined for displaying the error. 
	 */
	@Override
	protected void onBeforeRender()
	{
		super.onBeforeRender();
		// Get the messages for the current container. All direct and indirect children are checked.
		List<FeedbackMessage> list = Session.get().getFeedbackMessages().messages(getMessagesFilter());
		
		// filters the ignored components from 
		list = filterMsgsFromIgnoredComponents(list);
		final boolean showError = list.size() != 0;
		int count = 0;
		// visits all children and adds the error
		visitChildren(Component.class,
				new Component.IVisitor<Component>()
				{
					public Object component(Component comp)
					{
						// if there are no particular component ids defined, add the 
						// error attribute to all found components and child components.
						if (errorFeedbackComponentsIds.size()== 0) {
							// if the given component is not excluded for giving a feedback
							if (!feedbackExcludedComponentIds.contains(comp.getId()) ||
									feedbackExcludedComponentIds.size() == 0) { 
								addErrorAttribute(comp, showError);
							}
						} else {
							// if errors should be shown only to particular components
							// test whether the found component belongs to this set.
							if (errorFeedbackComponentsIds.contains(comp.getId())) { 
								// if the given component is not excluded for giving a feedback
								if (!feedbackExcludedComponentIds.contains(comp.getId()) ||
										feedbackExcludedComponentIds.size() == 0) {
										addErrorAttribute(comp, showError);
								}
							}
						}
						return CONTINUE_TRAVERSAL;
					}
				});
	}
	
	/**
	 * Filters the feedback messages originating from components whose id is contained 
	 * in the list of ignored or checked component ids.
	 * @param list The feedback message list that will be filtered.
	 * @return Returns the filtered feedback messages list.
	 */
	protected List<FeedbackMessage> filterMsgsFromIgnoredComponents(List<FeedbackMessage> list) {
		if (this.ignoredComponentIds.size() > 0) {
			int count = 0;
			while (count < list.size()) {
				if (ignoredComponentIds.contains(list.get(count).getReporter().getId())) {
					list.remove(count);
				} else {
					count++;
				}
			}
		} else if (this.checkedComponentIds.size() > 0) {			
			int count = 0;
			while (count < list.size()) {
				// if the current component id is not in the list of the checked component ids
				// remove it.
				if (!checkedComponentIds.contains(list.get(count).getReporter().getId())) {
					list.remove(count);
				} else {
					count++;
				}
			}
		}
		return list;
	}
	
	/**
	 * Adds/removes the CSS class attribute to/from the given component.
	 * @param comp The component for which the error is switched on/off.
	 * @param showError Indicates whether the feedback is switched on or off.
	 */
	protected void addErrorAttribute(Component comp, boolean showError) {		

		if (comp != null && !showError) {
			AttributeModifier am = findAttributeModifier(comp);
			if (am != null) comp.remove(am);
		}
													
		if (showError && comp != null) 
			  comp.add(new AttributeModifier("class", true, new Model(cssClassName)));		
	}
	
	/**
	 * Adds the id of a component that will show an validation error feedback.
	 * @param id The id of the component that will show a validation error feedback.
	 */
	public void addErrorFeedbackComponentId(String id) {
		errorFeedbackComponentsIds.add(id);
	}
	
	/**
	 * Adds the id of a component that will show an validation error feedback.
	 * @param ids An array of String ids of components that will show a validation error feedback.
	 */
	public void addErrorFeedbackComponetIds(String[] ids) {
		for (String i : ids) {
			this.errorFeedbackComponentsIds.add(i);
		}
	}
	
	/**
	 * Removes the id of component from the list of components that will visualize an error feedback.
	 * @param id The id to remove.
	 */
	public void removeErrorFeedbackComponentIds(String id) {
		errorFeedbackComponentsIds.remove(id);
	}
	
	/**
	 * Removes all ids of components that will show an error feedback.
	 */
	public void clearErrorFeedbackComponentIds() {
		errorFeedbackComponentsIds.clear();
	}
	
	/**
	 * Returns the number of ids of components that will show the error feedback.
	 * @return Returns the number of elements of elements that show the error feedback.
	 */
	public int getErrorFeedbackComponentIdsSize() {
		return errorFeedbackComponentsIds.size();
	}
	
	/**
	 * Adds the id of a component that will be ignored as a source of validation errors.
	 * @param id The id of the component. 
	 */
	public void addIgnoredComponentId(String id) {		
		ignoredComponentIds.add(id);
	}
	
	/**
	 * Adds the ids of a set of components that will be ignored  as a source of validation errors.
	 * @param ids An array of String objects representing ids of components.
	 */
	public void addIgnoredComponentIds(String[] ids) {
		for (String i : ids) {
			this.ignoredComponentIds.add(i);
		}
	}
	
	/**
	 * Removes the id of component from the list of components ignored as a soruce of validation error.
	 * @param id The id to remove.
	 */
	public void removeIgnoredComponentIds(String id) {
		ignoredComponentIds.remove(id);
	}
	
	/**
	 * Removes all defined ids of components that will be ignored as a source of validation error.
	 */
	public void clearIgnoredComponentIds() {
		ignoredComponentIds.clear();
	}
	
	/**
	 * Returns the number of ids of components that will be will be ignored as a source of validation error.
	 * @return Returns the number of elements.
	 */
	public int getIgnoredComponentIdsSize() {
		return ignoredComponentIds.size();
	}

	/**
	 * Adds the id of a component that will be checked as a source of validation error.
	 * @param id The id of the component.
	 */
	public void addCheckedComponentId(String id) {
		ignoredComponentIds.add(id);
	}
	
	/**
	 * Adds the ids of a set of components that will be checked as a source of validation error.
	 * @param ids An array of String objects representing ids of components.
	 */
	public void addCheckedComponentIds(String[] ids) {
		for (String i : ids) {
			this.ignoredComponentIds.add(i);
		}
	}
	
	/**
	 * Removes the id from the set of components  that will be checked as a source of validation error.
	 * @param id The id to remove.
	 */
	public void removeCheckedComponentIds(String id) {
		ignoredComponentIds.remove(id);
	}
	
	/**
	 * Removes all defined ids from the set of components that will be checked as a source of validation error.
	 */
	public void clearCheckedComponentIds() {
		ignoredComponentIds.clear();
	}
	
	/**
	 * Returns the number of ids of components that will be checked as a source of validation error.
	 * @return Returns the number of elements.
	 */
	public int getCheckedComponentIdsSize() {
		return ignoredComponentIds.size();
	}
	
	/**
	 * Adds the id of a component that will not visualize a feedback error.
	 * @param id The id of the component that will not show a feedback message.
	 */
	public void addFeedbackExcludedComponentId(String id) {
		this.feedbackExcludedComponentIds.add(id);
	}
	
	/**
	 * Adds the id of a set of components that  will not visualize a feedback error.
	 * @param id An array containing the ids of the components that will not show a feedback message.
	 */
	public void addFeedbackExcludedComponentIds(String[] ids) {
		for (String i : ids) {
			this.feedbackExcludedComponentIds.add(i);
		}
	}
	
	/**
	 * Removes the id from the set of components that will not visualize a feedback error.
	 * @param id The id to remove.
	 */
	public void removeFeedbackExcludedComponentIds(String id) {
		this.feedbackExcludedComponentIds.remove(id);
	}
	
	/**
	 * Removes all defined ids of components that will not show an error feedback.
	 */
	public void clearFeedbackExcludedComponentIds() {
		feedbackExcludedComponentIds.clear();
	}
	
	/**
	 * Returns the number of ids of components that will not visualize a feedback error.
	 * @return Returns the number of elements.
	 */
	public int getFeedbackExcludedComponentIdsSize() {
		return feedbackExcludedComponentIds.size();
	}

	/**
	 * Finds the CSS attribute modifier showing the error, if defined and returns it.
	 * @param comp The component where to search for the attribute modifier.
	 * @return Returns the CSS attribute modifier.
	 */
	private AttributeModifier findAttributeModifier(Component comp) {
		AttributeModifier retVal = null;
		List<IBehavior> list = comp.getBehaviors();
		int count = 0;
		while (count < list.size()) {
			IBehavior b = list.get(count++);
			if (b instanceof AttributeModifier &&
					((AttributeModifier)b).getAttribute().equals("class")) {
				retVal = ((AttributeModifier)b);
			}			
		}
		return retVal;		
	}

	/**
	 * @return Let subclass specify some other filter
	 */
	protected IFeedbackMessageFilter getMessagesFilter()
	{
		return new ContainerFeedbackMessageFilter(this);
	}
}

